#ifndef _lf_myri_packet_h_
#define _lf_myri_packet_h_
/*
 * Generic Myrinet utility packets
 */

#define MYRI_PACKET_TYPE 0x4004
#define LF_MAX_MYRI_RAW_PKT_LEN (1024 - MYRI_MAX_ROUTE_LEN)

#define MYRI_NIC_SCOUT_OPAQUE_SIZE 20
#define MYRI_NIC_REPLY_OPAQUE_SIZE 256

enum myri_packet_subtype {
  MYRI_SUBTYPE_NIC_SCOUT,
  MYRI_SUBTYPE_NIC_SCOUT_REPLY,
  MYRI_SUBTYPE_NO_OP,
  MYRI_SUBTYPE_COUNT
};

#define MYRI_PROTOCOL_ID 7


/*
 * Header common to all Myrinet packets
 */
struct lf_myri_packet_hdr {
  uint16_t type_16;		/* MYRI_PACKET_TYPE */
  uint16_t subtype_16;		/* MYRI subtype */
};

struct lf_myri_packet {

  struct lf_myri_packet_hdr hdr;

  union {

    struct myri_query {	/* subtype = MYRI_QUERY */
      uint8_t reply_route_len;
      uint8_t reply_route[MYRI_MAX_ROUTE_LEN];
    } query;

    struct myri_query_reply {	/* subtype = MYRI_QUERY_REPLY */
      uint16_t num_items;	/* number of items in the reply */
      uint8_t num_reply_packets; /* total number of reply packets */
      uint8_t reply_packets_seq; /* index of this reply packet */
      uint8_t data[1];		/* data of reply, see below for format */
    } query_reply;

    struct myri_set_route {  /* subtype = MYRI_SET_ROUTE */
      uint8_t mac_addr[6];	/* MAC address of destination */
      uint8_t destport;		/* port on destination */
      uint8_t srcport;		/* source port for receiving NIC */
      uint8_t route_len;
      uint8_t route[MYRI_MAX_ROUTE_LEN];

      uint8_t reply_route_len;
      uint8_t reply_route[MYRI_MAX_ROUTE_LEN];
    } set_route;
    
    struct myri_set_route_reply {  /* subtype = MYRI_SET_ROUTE_REPLY */
      uint8_t return_code;	/* 0 for success, non-zero otherwise,
			           firmware specific interpretation */
    } set_route_reply;


    struct myri_pause_sending {	/* subtype = MYRI_PAUSE_SENDING */
      uint8_t route_len;
      uint8_t route[MYRI_MAX_ROUTE_LEN];
    } pause_sending;
      
    struct myri_pause_sending_reply { /* subtype = MYRI_PAUSE_SENDING_REPLY */
      uint32_t pad;			/* make structure non-empty */
    } pause_sending_reply;

    struct myri_resume_sending {	/* subtype = MYRI_RESUME_SENDING */
      uint8_t route_len;
      uint8_t route[MYRI_MAX_ROUTE_LEN];
    } resume_sending;
      
    struct myri_resume_sending_reply { /* subtype = MYRI_RESUME_SENDING_REPLY */
      uint32_t pad;			/* make structure non-empty */
    } resume_sending_reply;

    /* forward "data" out onto the wire */
    struct myri_forward {	/* subtype = MYRI_FORWARD */
      uint8_t port;		/* local port to use */
      uint16_t data_len;
      uint8_t data[1];		
    } forward;
    
  } u;
};

/*
 * A NIC scout packet that everything should be able to reply to
 */
struct myri_nic_scout {	/* subtype = NIC_SCOUT */
  struct lf_myri_packet_hdr hdr;  /* subtype = MYRI_SUBTYPE_NIC_SCOUT */

  /* the following blob of data is returned unchanged by the responder
   * in the NIC reply.
   */
  uint8_t opaque_scout_data[MYRI_NIC_SCOUT_OPAQUE_SIZE];

  uint8_t reply_route[MYRI_MAX_ROUTE_LEN/2];
  uint8_t reply_route_len_8;
  uint8_t nic_replied_8;	/* set by NIC if auto-reply done */
};

/*
 * The reply to a generic NIC scout
 */
struct myri_nic_scout_reply { /* subtype = NIC_SCOUT_REPLY */
  struct lf_myri_packet_hdr hdr;  /* subtype = MYRI_SUBTYPE_NIC_SCOUT_REPLY */

  /* opaque data used only by higher levels */
  uint8_t opaque_scout_data[MYRI_NIC_SCOUT_OPAQUE_SIZE];
  uint8_t opaque_reply_data[MYRI_NIC_REPLY_OPAQUE_SIZE];

  uint8_t mac_addr[6];	/* MAC address of packet sender */
  uint8_t port_8;		/* port scout was received on */
  uint8_t num_ports_8;	/* number of ports on NIC */
  uint8_t firmware_type_8;	/* GM, MX, XM, etc. */
};

/* 
 * Format of MYRI_QUERY_REPLY packet.
 * The data of an MYRI_QUERY_REPLY is formatted in triplets consisting of
 * a type, a length, and arbitrary data associated with the type.  
 * An MYRI_QUERY reply may generate multiple MYRI_QUERY_REPLY packets due
 * to MTU limitations, so the reply structure allows for multi-packet
 * replies.
 *
 * The header of the QUERY_REPLY specifies the number of triplets in this
 * packet (num_items), also the number of reply packets (num_reply_packets),
 * and the number of this reply packet in:
 *  (0 <= reply_packet_seq < num_reply_packets)
 *
 * Each triplet has the format below:
 */
struct myri_triplet {
  uint16_t length;	/* total length in bytes including lenght&data_id */
  uint16_t data_id;	/* data ID of this triplet, data format is implicit */
  uint8_t data[1];	/* the data associated with the ID */
};

enum myri_query_data_id {
  MYRI_QUERY_REPLY_NIC_TYPE,
  MYRI_QUERY_REPLY_NIC_SERIAL_NUMBER,
  MYRI_QUERY_REPLY_FIRMWARE_SPECIFIC_DATA,
  MYRI_QUERY_REPLY_
};

/*
 * Convenience structures for triplets
 */
struct myri_triplet_nic_type {
  uint16_t length;		/* MYRI_NIC_TYPE_LEN + 4 */
  uint16_t data_id;		/* MYRI_QUERY_REPLY_NIC_TYPE */
  uint8_t nic_type[LF_SHORT_STRING_LEN];  /* null-terminated string */
};

struct myri_triplet_nic_serial {
  uint16_t length;		/* MYRI_NIC_SERIAL_LEN + 4 */
  uint16_t data_id;		/* MYRI_QUERY_REPLY_NIC_SERIAL_NUMBER */
  uint8_t nic_type[LF_SHORT_STRING_LEN];  /* null-terminated string */
};

struct myri_firmware_specific_data {
  uint16_t length;
  uint16_t data_id;		/* MYRI_QUERY_REPLY_FIRMWARE_SPECIFIC_DATA */
  uint8_t data[1];
};

#endif /* _lf_myri_packet_h_ */
